// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

document.addEventListener('DOMContentLoaded', function() {
    // 인증 확인
    const token = localStorage.getItem('token');
    if (!token) {
        window.location.href = './login.html';
        return;
    }
    
    // UI 요소
    const userNameElement = document.getElementById('user-name');
    const userAvatarElement = document.getElementById('user-avatar');
    const logoutBtn = document.getElementById('logout-btn');
    const themeToggle = document.getElementById('theme-toggle');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.getElementById('sidebar');
    
    // 지표 관련 요소
    const addMetricForm = document.getElementById('add-metric-form');
    const metricTypeSelect = document.getElementById('metric-type');
    const chartMetricTypeSelect = document.getElementById('chart-metric-type');
    const chartTimeRangeSelect = document.getElementById('chart-time-range');
    const historyMetricTypeSelect = document.getElementById('history-metric-type');
    const historySortSelect = document.getElementById('history-sort');
    const metricsSummary = document.getElementById('metrics-summary');
    const metricsTable = document.getElementById('metrics-table');
    const metricsTbody = document.getElementById('metrics-tbody');
    
    // 모달 요소
    const editModal = document.getElementById('edit-modal');
    const editForm = document.getElementById('edit-metric-form');
    const closeModalBtn = document.getElementById('close-modal');
    
    // 사용자 정보 로드
    let user = JSON.parse(localStorage.getItem('user')) || {};
    
    // 차트 관련 변수
    let metricsChart = null;
    
    // 사용자 정보 표시
    userNameElement.textContent = user.firstName ? `${user.firstName} ${user.lastName || ''}` : user.username;
    if (user.firstName && user.lastName) {
        userAvatarElement.textContent = `${user.firstName.charAt(0)}${user.lastName.charAt(0)}`;
    } else if (user.username) {
        userAvatarElement.textContent = user.username.substring(0, 2).toUpperCase();
    }
    
    // 다크 모드 적용
    const currentTheme = localStorage.getItem('theme');
    if (currentTheme === 'dark') {
        document.body.setAttribute('data-theme', 'dark');
        themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
    }
    
    // Sidebar toggle on mobile
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    });
    
    // Theme toggle
    themeToggle.addEventListener('click', function() {
        if (document.body.getAttribute('data-theme') === 'dark') {
            document.body.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
        } else {
            document.body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        }
    });
    
    // Logout
    logoutBtn.addEventListener('click', function() {
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        window.location.href = './login.html';
    });
    
    // 측정 날짜 기본값 설정 (현재 시간)
    const now = new Date();
    const formattedNow = now.toISOString().slice(0, 16);
    document.getElementById('measured-date').value = formattedNow;
    
    // 지표 유형 변경 시 입력 형식 업데이트
    metricTypeSelect.addEventListener('change', function() {
        const valueContainer = document.getElementById('value-container');
        const metricValue = document.getElementById('metric-value');
        
        switch(this.value) {
            case 'blood_pressure':
                valueContainer.querySelector('label').textContent = '혈압 (수축기/이완기)';
                metricValue.placeholder = '예: 120/80';
                break;
            case 'weight':
                valueContainer.querySelector('label').textContent = '체중 (kg)';
                metricValue.placeholder = '예: 75';
                break;
            case 'heart_rate':
                valueContainer.querySelector('label').textContent = '심박수 (BPM)';
                metricValue.placeholder = '예: 72';
                break;
            default:
                valueContainer.querySelector('label').textContent = '측정값';
                metricValue.placeholder = '';
        }
    });
    
    // 새 지표 추가 폼 제출
    addMetricForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const metricType = metricTypeSelect.value;
        let value = document.getElementById('metric-value').value.trim();
        const measuredAt = document.getElementById('measured-date').value;
        const notes = document.getElementById('notes').value.trim();
        
        // 입력 유효성 검사
        if (!metricType || !value || !measuredAt) {
            alert('모든 필수 필드를 채워주세요.');
            return;
        }
        
        try {
            // 지표 데이터 생성
            const metricData = {
                metricType: metricType,
                value: value,
                measuredAt: new Date(measuredAt).toISOString(),
                notes: notes || null
            };
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/health-metrics`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify(metricData)
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '지표 저장 중 오류가 발생했습니다');
            }
            
            // 폼 초기화
            addMetricForm.reset();
            document.getElementById('measured-date').value = formattedNow;
            
            // 성공 메시지
            alert('건강 지표가 성공적으로 저장되었습니다.');
            
            // 데이터 다시 로드
            loadMetricsSummary();
            loadMetricsHistory();
            updateChart();
            
        } catch (error) {
            console.error('Error saving metric:', error);
            alert(error.message);
        }
    });
    
    // 필터 변경 이벤트
    chartMetricTypeSelect.addEventListener('change', updateChart);
    chartTimeRangeSelect.addEventListener('change', updateChart);
    historyMetricTypeSelect.addEventListener('change', loadMetricsHistory);
    historySortSelect.addEventListener('change', loadMetricsHistory);
    
    // 모달 닫기
    closeModalBtn.addEventListener('click', function() {
        editModal.style.display = 'none';
    });
    
    // 모달 외부 클릭 시 닫기
    window.addEventListener('click', function(event) {
        if (event.target === editModal) {
            editModal.style.display = 'none';
        }
    });
    
    // 지표 수정 폼 제출
    editForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const id = document.getElementById('edit-id').value;
        const value = document.getElementById('edit-value').value.trim();
        const measuredAt = document.getElementById('edit-date').value;
        const notes = document.getElementById('edit-notes').value.trim();
        
        try {
            // 지표 데이터 생성
            const metricData = {
                value: value,
                measuredAt: new Date(measuredAt).toISOString(),
                notes: notes || null
            };
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/health-metrics/${id}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify(metricData)
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '지표 수정 중 오류가 발생했습니다');
            }
            
            // 모달 닫기
            editModal.style.display = 'none';
            
            // 성공 메시지
            alert('건강 지표가 성공적으로 수정되었습니다.');
            
            // 데이터 다시 로드
            loadMetricsSummary();
            loadMetricsHistory();
            updateChart();
            
        } catch (error) {
            console.error('Error updating metric:', error);
            alert(error.message);
        }
    });
    
    // 초기 데이터 로드
    loadMetricsSummary();
    loadMetricsHistory();
    initChart();
    
    // 최근 지표 요약 로드
    async function loadMetricsSummary() {
        try {
            const metricsContainer = document.getElementById('metrics-summary');
            metricsContainer.innerHTML = '<div class="loading"><div class="spinner"></div></div>';
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/health-metrics/summary`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('지표 요약을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const summary = data.data || {};
            
            let html = '';
            
            // 혈압 요약
            if (summary.blood_pressure) {
                html += createSummaryCard('혈압', summary.blood_pressure.value, summary.blood_pressure.measured_at, 'fa-heartbeat', getBloodPressureStatus(summary.blood_pressure.value));
            } else {
                html += createEmptyCard('혈압', 'fa-heartbeat');
            }
            
            // 체중 요약
            if (summary.weight) {
                html += createSummaryCard('체중', summary.weight.value + ' kg', summary.weight.measured_at, 'fa-weight');
            } else {
                html += createEmptyCard('체중', 'fa-weight');
            }
            
            // 심박수 요약
            if (summary.heart_rate) {
                html += createSummaryCard('심박수', summary.heart_rate.value + ' BPM', summary.heart_rate.measured_at, 'fa-heartbeat', getHeartRateStatus(summary.heart_rate.value));
            } else {
                html += createEmptyCard('심박수', 'fa-heartbeat');
            }
            
            metricsContainer.innerHTML = html;
            
        } catch (error) {
            console.error('Error loading metrics summary:', error);
            document.getElementById('metrics-summary').innerHTML = `
                <div class="alert alert-error">
                    <p>${error.message || '지표를 불러오는 중 오류가 발생했습니다.'}</p>
                </div>
            `;
        }
    }
    
    // 지표 기록 로드
    async function loadMetricsHistory() {
        try {
            const metricType = historyMetricTypeSelect.value;
            const sortOrder = historySortSelect.value;
            
            const tbody = document.getElementById('metrics-tbody');
            tbody.innerHTML = '<tr><td colspan="5" class="loading"><div class="spinner"></div></td></tr>';
            
            // API URL 구성
            let url = `${API_BASE_URL}/api/health-metrics?sort=${sortOrder}`;
            if (metricType !== 'all') {
                url += `&type=${metricType}`;
            }
            
            // API 요청
            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('지표 기록을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const metrics = data.data || [];
            
            if (metrics.length === 0) {
                tbody.innerHTML = '<tr><td colspan="5" style="text-align:center;">기록된 지표가 없습니다.</td></tr>';
                return;
            }
            
            let html = '';
            
            metrics.forEach(metric => {
                const date = new Date(metric.measured_at);
                const formattedDate = formatDateTime(date);
                
                let typeText, valueText;
                
                switch(metric.metric_type) {
                    case 'blood_pressure':
                        typeText = '혈압';
                        valueText = metric.value;
                        break;
                    case 'weight':
                        typeText = '체중';
                        valueText = metric.value + ' kg';
                        break;
                    case 'heart_rate':
                        typeText = '심박수';
                        valueText = metric.value + ' BPM';
                        break;
                    default:
                        typeText = metric.metric_type;
                        valueText = metric.value;
                }
                
                html += `
                    <tr>
                        <td>${typeText}</td>
                        <td>${valueText}</td>
                        <td>${formattedDate}</td>
                        <td>${metric.notes || '-'}</td>
                        <td class="actions">
                            <button class="edit" onclick="editMetric(${metric.id})">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="delete" onclick="deleteMetric(${metric.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </td>
                    </tr>
                `;
            });
            
            tbody.innerHTML = html;
            
        } catch (error) {
            console.error('Error loading metrics history:', error);
            document.getElementById('metrics-tbody').innerHTML = `
                <tr>
                    <td colspan="5" class="alert alert-error">
                        ${error.message || '지표 기록을 불러오는 중 오류가 발생했습니다.'}
                    </td>
                </tr>
            `;
        }
    }
    
    // 차트 초기화
// Chart oluşturma fonksiyonu düzeltilmiş versiyonu
function initChart() {
    // Canvas elementini bul
    const canvas = document.getElementById('metrics-chart');
    if (!canvas) {
        console.error("Canvas element 'metrics-chart' bulunamadı!");
        return;
    }
    
    // Canvas context'ini kontrol et
    const ctx = canvas.getContext('2d');
    if (!ctx) {
        console.error("Canvas context alınamadı!");
        return;
    }
    
    // Mevcut bir chart varsa yok et
    if (metricsChart) {
        metricsChart.destroy();
    }
    
    // Yeni chart oluştur
    metricsChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: [],
            datasets: [{
                label: '',
                data: [],
                backgroundColor: 'rgba(106, 170, 228, 0.2)',
                borderColor: 'rgba(106, 170, 228, 1)',
                borderWidth: 2,
                tension: 0.1,
                fill: true
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: ''
                },
                tooltip: {
                    mode: 'index',
                    intersect: false
                }
            },
            scales: {
                x: {
                    display: true,
                    title: {
                        display: true,
                        text: '날짜'
                    }
                },
                y: {
                    display: true,
                    title: {
                        display: true,
                        text: ''
                    }
                }
            }
        }
    });
    
    // İlk veri yüklemesini yap
    updateChart();
}
    
    // 차트 업데이트
    async function updateChart() {
    try {
        // Loading göster
        const loadingElement = document.getElementById('chart-loading');
        if (loadingElement) {
            loadingElement.style.display = 'flex';
        }
        
        const metricType = document.getElementById('chart-metric-type').value;
        const timeRange = document.getElementById('chart-time-range').value;
        
        // API URL yapılandırması
        let url = `${API_BASE_URL}/api/health-metrics?type=${metricType}`;
        
        if (timeRange !== 'all') {
            const startDate = new Date();
            startDate.setDate(startDate.getDate() - parseInt(timeRange));
            url += `&from=${startDate.toISOString()}`;
        }
        
        // API isteği
        const response = await fetch(url, {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('token')}`
            }
        });
        
        if (!response.ok) {
            throw new Error('차트 데이터를 불러오는 중 오류가 발생했습니다');
        }
        
        const data = await response.json();
        const metrics = data.data || [];
        
        // Chart varsa güncelle
        if (metricsChart) {
            // Zaman sırasına göre sırala
            metrics.sort((a, b) => new Date(a.measured_at) - new Date(b.measured_at));
            
            // Chart verilerini hazırla
            const labels = [];
            const values = [];
            
            metrics.forEach(metric => {
                const date = new Date(metric.measured_at);
                labels.push(formatDate(date));
                
                // Kan basıncı için sistolik değeri kullan
                if (metric.metric_type === 'blood_pressure' && metric.value.includes('/')) {
                    const parts = metric.value.split('/');
                    values.push(parseInt(parts[0]));
                } else {
                    values.push(parseFloat(metric.value));
                }
            });
            
            // Y ekseni etiketi ve başlık ayarla
            let title, yAxisLabel;
            switch(metricType) {
                case 'blood_pressure':
                    title = '혈압 추이';
                    yAxisLabel = '수축기 혈압 (mmHg)';
                    break;
                case 'weight':
                    title = '체중 추이';
                    yAxisLabel = '체중 (kg)';
                    break;
                case 'heart_rate':
                    title = '심박수 추이';
                    yAxisLabel = '심박수 (BPM)';
                    break;
            }
            
            // Chart'ı güncelle
            metricsChart.data.labels = labels;
            metricsChart.data.datasets[0].data = values;
            metricsChart.data.datasets[0].label = title;
            metricsChart.options.plugins.title.text = title;
            metricsChart.options.scales.y.title.text = yAxisLabel;
            metricsChart.update();
        }
        
        // Loading gizle
        if (loadingElement) {
            loadingElement.style.display = 'none';
        }
        
    } catch (error) {
        console.error('Error updating chart:', error);
        // Loading gizle
        const loadingElement = document.getElementById('chart-loading');
        if (loadingElement) {
            loadingElement.style.display = 'none';
        }
    }
}
    
    // 요약 카드 생성 함수
    function createSummaryCard(title, value, dateStr, icon, status = '') {
        const date = new Date(dateStr);
        const formattedDate = formatDateTime(date);
        
        let statusClass = '';
        let statusText = '';
        
        if (status) {
            switch(status) {
                case 'normal':
                    statusClass = 'text-success';
                    statusText = '정상';
                    break;
                case 'elevated':
                    statusClass = 'text-warning';
                    statusText = '주의';
                    break;
                case 'high':
                    statusClass = 'text-danger';
                    statusText = '높음';
                    break;
                case 'low':
                    statusClass = 'text-info';
                    statusText = '낮음';
                    break;
            }
        }
        
        return `
            <div class="metric-card">
                <div class="metric-header">
                    <div class="metric-title">
                        <i class="fas ${icon}"></i> ${title}
                    </div>
                    ${statusText ? `<span class="${statusClass}">${statusText}</span>` : ''}
                </div>
                <div class="metric-value">${value}</div>
                <div class="metric-date">
                    <i class="far fa-clock"></i> ${formattedDate}
                </div>
            </div>
        `;
    }
    
    // 빈 카드 생성 함수
    function createEmptyCard(title, icon) {
        return `
            <div class="metric-card">
                <div class="metric-header">
                    <div class="metric-title">
                        <i class="fas ${icon}"></i> ${title}
                    </div>
                </div>
                <div class="metric-value">기록 없음</div>
                <div class="metric-date">
                    <a href="#add-metric-form">새 값 추가</a>
                </div>
            </div>
        `;
    }
    
    // 혈압 상태 평가
    function getBloodPressureStatus(bloodPressure) {
        // Format: "120/80"
        if (!bloodPressure || !bloodPressure.includes('/')) return '';
        
        const [systolic, diastolic] = bloodPressure.split('/').map(Number);
        
        // 정상: 수축기 < 120 AND 이완기 < 80
        if (systolic < 120 && diastolic < 80) return 'normal';
        
        // 주의: 수축기 120-129 AND 이완기 < 80
        if (systolic >= 120 && systolic <= 129 && diastolic < 80) return 'elevated';
        
        // 고혈압: 수축기 >= 130 OR 이완기 >= 80
        return 'high';
    }
    
    // 심박수 상태 평가
    function getHeartRateStatus(heartRate) {
        const hr = Number(heartRate);
        
        // 서맥: < 60 BPM
        if (hr < 60) return 'low';
        
        // 정상: 60-100 BPM
        if (hr >= 60 && hr <= 100) return 'normal';
        
        // 빈맥: > 100 BPM
        return 'high';
    }
    
    // 날짜 형식 함수 (YYYY-MM-DD)
    function formatDate(date) {
        return `${date.getFullYear()}-${String(date.getMonth() + 1).padStart(2, '0')}-${String(date.getDate()).padStart(2, '0')}`;
    }
    
    // 날짜 및 시간 형식 함수
    function formatDateTime(date) {
        return `${date.getFullYear()}-${String(date.getMonth() + 1).padStart(2, '0')}-${String(date.getDate()).padStart(2, '0')} ${String(date.getHours()).padStart(2, '0')}:${String(date.getMinutes()).padStart(2, '0')}`;
    }
    
    // 전역 스코프에 함수 등록
    window.editMetric = function(id) {
        openEditModal(id);
    };
    
    window.deleteMetric = function(id) {
        confirmDeleteMetric(id);
    };
});

// 지표 수정 모달 열기
async function openEditModal(id) {
    const token = localStorage.getItem('token');
    
    try {
        // API 요청
        const response = await fetch(`${API_BASE_URL}/api/health-metrics/${id}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (!response.ok) {
            throw new Error('지표 정보를 불러오는 중 오류가 발생했습니다');
        }
        
        const data = await response.json();
        const metric = data.data;
        
        // 모달 폼 채우기
        document.getElementById('edit-id').value = metric.id;
        document.getElementById('edit-type').value = metric.metric_type;
        document.getElementById('edit-value').value = metric.value;
        
        // 측정 날짜 변환
        const measuredDate = new Date(metric.measured_at);
        const formattedDate = measuredDate.toISOString().slice(0, 16);
        document.getElementById('edit-date').value = formattedDate;
        
        document.getElementById('edit-notes').value = metric.notes || '';
        
        // 모달 표시
        document.getElementById('edit-modal').style.display = 'block';
        
    } catch (error) {
        console.error('Error loading metric for edit:', error);
        alert(error.message);
    }
}

// 지표 삭제 확인
function confirmDeleteMetric(id) {
    if (confirm('이 측정값을 삭제하시겠습니까? 이 작업은 취소할 수 없습니다.')) {
        deleteMetric(id);
    }
}

// 지표 삭제 함수
async function deleteMetric(id) {
    const token = localStorage.getItem('token');
    
    try {
        // API 요청
        const response = await fetch(`${API_BASE_URL}/api/health-metrics/${id}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (!response.ok) {
            throw new Error('지표를 삭제하는 중 오류가 발생했습니다');
        }
        
        // 성공 메시지
        alert('측정값이 성공적으로 삭제되었습니다.');
        
        // 데이터 다시 로드
        loadMetricsSummary();
        loadMetricsHistory();
        updateChart();
        
    } catch (error) {
        console.error('Error deleting metric:', error);
        alert(error.message);
    }
}

// loadMetricsSummary, loadMetricsHistory, updateChart 함수를 전역 스코프에 정의
function loadMetricsSummary() {
    // DOM 로드 이후에 호출되는 함수를 전역에서 호출하기 위한 래퍼
    const metricsContainer = document.getElementById('metrics-summary');
    if (!metricsContainer) return;
    
    // 이미 DOM 로드 이벤트 내에 정의된 함수를 직접 호출할 수 없으므로
    // 이벤트를 트리거하여 해당 함수가 호출되도록 함
    const event = new Event('refresh-metrics-summary');
    document.dispatchEvent(event);
}

function loadMetricsHistory() {
    // loadMetricsSummary와 같은 패턴으로 DOM 이벤트를 통해 간접 호출
    const event = new Event('refresh-metrics-history');
    document.dispatchEvent(event);
}

function updateChart() {
    // loadMetricsSummary와 같은 패턴으로 DOM 이벤트를 통해 간접 호출
    const event = new Event('refresh-metrics-chart');
    document.dispatchEvent(event);
}

// DOM 이벤트 리스너 추가
document.addEventListener('refresh-metrics-summary', function() {
    // 페이지가 이미 로드된 후, DOM 이벤트 핸들러에서 원본 함수를 호출
    const loadFunction = document.querySelector('[data-function="loadMetricsSummary"]');
    if (loadFunction && typeof loadFunction.onclick === 'function') {
        loadFunction.onclick();
    }
});

document.addEventListener('refresh-metrics-history', function() {
    const loadFunction = document.querySelector('[data-function="loadMetricsHistory"]');
    if (loadFunction && typeof loadFunction.onclick === 'function') {
        loadFunction.onclick();
    }
});

document.addEventListener('refresh-metrics-chart', function() {
    const loadFunction = document.querySelector('[data-function="updateChart"]');
    if (loadFunction && typeof loadFunction.onclick === 'function') {
        loadFunction.onclick();
    }
});