// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

document.addEventListener('DOMContentLoaded', function() {
    // 인증 확인
    const token = localStorage.getItem('token');
    if (!token) {
        window.location.href = './login.html';
        return;
    }
    
    // 사용자 정보 로드
    let user = JSON.parse(localStorage.getItem('user')) || {};
    
    // UI 요소
    const userNameElement = document.getElementById('user-name');
    const userAvatarElement = document.getElementById('user-avatar');
    const profilePictureElement = document.getElementById('profile-picture');
    const logoutBtn = document.getElementById('logout-btn');
    const themeToggle = document.getElementById('theme-toggle');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.getElementById('sidebar');
    
    // 양식 요소
    const profileForm = document.getElementById('profile-form');
    const passwordForm = document.getElementById('password-form');
    const deleteConfirmationForm = document.getElementById('delete-confirmation-form');
    
    // 토글 요소
    const darkModeToggle = document.getElementById('darkModeToggle');
    const notificationsToggle = document.getElementById('notificationsToggle');
    const serverTtsToggle = document.getElementById('serverTtsToggle');
    
    // 슬라이더 요소
    const speechRateSlider = document.getElementById('speechRate');
    const speechRateValue = document.getElementById('speechRateValue');
    const speechPitchSlider = document.getElementById('speechPitch');
    const speechPitchValue = document.getElementById('speechPitchValue');
    
    // 모달 요소
    const passwordSuccessModal = document.getElementById('password-success-modal');
    const closePasswordModal = document.getElementById('close-password-modal');
    const deactivateModal = document.getElementById('deactivate-modal');
    const deleteModal = document.getElementById('delete-modal');
    
    // 버튼
    const deactivateBtn = document.getElementById('deactivate-btn');
    const confirmDeactivateBtn = document.getElementById('confirm-deactivate');
    const cancelDeactivateBtn = document.getElementById('cancel-deactivate');
    const deleteAccountBtn = document.getElementById('delete-account-btn');
    const confirmDeleteBtn = document.getElementById('confirm-delete');
    const cancelDeleteBtn = document.getElementById('cancel-delete');
    const logoutAllBtn = document.getElementById('logout-all-btn');
    
    // 사용자 정보 표시
    updateUserInfo();
    
    // 프로필 양식 초기화
    initProfileForm();
    
    // 환경설정 로드
    loadPreferences();
    
    // IP 주소 가져오기
    fetchIPAddress();
    
    // 음성 목록 가져오기
    fetchVoices();
    
    // 언어 선택 이벤트 리스너
    setupLanguageSelection();
    
    // 탭 전환 설정
    setupTabSwitching();
    
    // 다크 모드 적용
    const currentTheme = localStorage.getItem('theme');
    if (currentTheme === 'dark') {
        document.body.setAttribute('data-theme', 'dark');
        themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        darkModeToggle.checked = true;
    }
    
    // Sidebar toggle on mobile
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    });
    
    // Theme toggle
    themeToggle.addEventListener('click', function() {
        toggleDarkMode();
    });
    
    // Logout
    logoutBtn.addEventListener('click', function() {
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        window.location.href = './login.html';
    });
    
    // Dark Mode Toggle
    darkModeToggle.addEventListener('change', function() {
        toggleDarkMode();
    });
    
    // Speech Rate Slider
    speechRateSlider.addEventListener('input', function() {
        speechRateValue.textContent = this.value;
        savePreferences();
    });
    
    // Speech Pitch Slider
    speechPitchSlider.addEventListener('input', function() {
        speechPitchValue.textContent = this.value;
        savePreferences();
    });
    
    // Save preferences when toggles change
    notificationsToggle.addEventListener('change', savePreferences);
    serverTtsToggle.addEventListener('change', savePreferences);
    
    // Profile Form Submit
    profileForm.addEventListener('submit', function(e) {
        e.preventDefault();
        updateProfile();
    });
    
    // Password Form Submit
    passwordForm.addEventListener('submit', function(e) {
        e.preventDefault();
        changePassword();
    });
    
    // Close Password Success Modal
    closePasswordModal.addEventListener('click', function() {
        passwordSuccessModal.style.display = 'none';
    });
    
    // Deactivate Account Button
    deactivateBtn.addEventListener('click', function() {
        deactivateModal.style.display = 'block';
    });
    
    // Cancel Deactivate
    cancelDeactivateBtn.addEventListener('click', function() {
        deactivateModal.style.display = 'none';
    });
    
    // Confirm Deactivate
    confirmDeactivateBtn.addEventListener('click', function() {
        deactivateAccount();
    });
    
    // Delete Account Button
    deleteAccountBtn.addEventListener('click', function() {
        deleteModal.style.display = 'block';
    });
    
    // Cancel Delete
    cancelDeleteBtn.addEventListener('click', function() {
        deleteModal.style.display = 'none';
    });
    
    // Confirm Delete (Form Submit)
    deleteConfirmationForm.addEventListener('submit', function(e) {
        e.preventDefault();
        deleteAccount();
    });
    
    // Logout from all devices
    logoutAllBtn.addEventListener('click', function() {
        logoutAllDevices();
    });
    
    // 사용자 정보 업데이트 함수
    function updateUserInfo() {
        userNameElement.textContent = user.firstName ? `${user.firstName} ${user.lastName || ''}` : user.username;
        
        // 아바타 이니셜
        let initials = '';
        if (user.firstName && user.lastName) {
            initials = `${user.firstName.charAt(0)}${user.lastName.charAt(0)}`;
        } else if (user.username) {
            initials = user.username.substring(0, 2).toUpperCase();
        }
        
        userAvatarElement.textContent = initials;
        profilePictureElement.textContent = initials;
    }
    
    // 프로필 양식 초기화
    function initProfileForm() {
        document.getElementById('firstName').value = user.firstName || '';
        document.getElementById('lastName').value = user.lastName || '';
        document.getElementById('email').value = user.email || '';
        document.getElementById('username').value = user.username || '';
    }
    
    // 사용자 환경설정 로드
    function loadPreferences() {
        const settings = user.settings || {
            darkMode: false,
            notifications: true,
            speechRate: 1.0,
            speechPitch: 1.0,
            useBackendTts: true,
            preferredLanguage: 'ko',
            preferredVoice: ''
        };
        
        // 설정 요소 값 설정
        darkModeToggle.checked = settings.darkMode;
        notificationsToggle.checked = settings.notifications;
        serverTtsToggle.checked = settings.useBackendTts;
        
        speechRateSlider.value = settings.speechRate;
        speechRateValue.textContent = settings.speechRate;
        
        speechPitchSlider.value = settings.speechPitch;
        speechPitchValue.textContent = settings.speechPitch;
        
        // 언어 선택 업데이트
        updateLanguageSelection(settings.preferredLanguage);
    }
    
    // 사용자 환경설정 저장
    function savePreferences() {
        const settings = {
            darkMode: darkModeToggle.checked,
            notifications: notificationsToggle.checked,
            useBackendTts: serverTtsToggle.checked,
            speechRate: parseFloat(speechRateSlider.value),
            speechPitch: parseFloat(speechPitchSlider.value),
            preferredLanguage: document.querySelector('.language-option.selected').dataset.lang,
            preferredVoice: document.querySelector('.voice-option.selected')?.dataset.voice || ''
        };
        
        updateUserSettings(settings);
    }
    
    // 다크 모드 전환
    function toggleDarkMode() {
        if (document.body.getAttribute('data-theme') === 'dark') {
            document.body.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
            darkModeToggle.checked = false;
        } else {
            document.body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
            darkModeToggle.checked = true;
        }
        
        // 환경설정 업데이트
        savePreferences();
    }
    
    // 탭 전환 설정
    function setupTabSwitching() {
        const tabs = document.querySelectorAll('.settings-tab');
        const panels = document.querySelectorAll('.settings-panel');
        
        tabs.forEach(tab => {
            tab.addEventListener('click', function() {
                // 모든 탭 비활성화
                tabs.forEach(t => t.classList.remove('active'));
                // 모든 패널 숨기기
                panels.forEach(p => p.classList.remove('active'));
                
                // 선택한 탭 활성화
                this.classList.add('active');
                
                // 해당 패널 표시
                const targetPanel = document.getElementById(`${this.dataset.tab}-panel`);
                targetPanel.classList.add('active');
            });
        });
    }
    
    // 언어 선택 설정
    function setupLanguageSelection() {
        const languageOptions = document.querySelectorAll('.language-option');
        
        languageOptions.forEach(option => {
            option.addEventListener('click', function() {
                // 모든 옵션 비활성화
                languageOptions.forEach(opt => opt.classList.remove('selected'));
                
                // 선택한 옵션 활성화
                this.classList.add('selected');
                
                // 환경설정 저장
                savePreferences();
            });
        });
    }
    
    // 언어 선택 업데이트
    function updateLanguageSelection(language) {
        const languageOptions = document.querySelectorAll('.language-option');
        
        languageOptions.forEach(option => {
            option.classList.remove('selected');
            if (option.dataset.lang === language) {
                option.classList.add('selected');
            }
        });
    }
    
    // IP 주소 가져오기
    async function fetchIPAddress() {
        try {
            const response = await fetch('https://api.ipify.org?format=json');
            if (response.ok) {
                const data = await response.json();
                document.getElementById('current-ip').textContent = data.ip;
            }
        } catch (error) {
            console.error('Error fetching IP address:', error);
            document.getElementById('current-ip').textContent = 'Unknown';
        }
    }
    
    // 음성 옵션 가져오기
    async function fetchVoices() {
        try {
            const voiceContainer = document.getElementById('voice-selection');
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/voices`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('Failed to fetch voices');
            }
            
            const voices = await response.json();
            
            // 로딩 제거
            voiceContainer.innerHTML = '';
            
            if (voices.length === 0) {
                voiceContainer.innerHTML = '<p>사용 가능한 음성이 없습니다.</p>';
                return;
            }
            
            // 사용자 선호 음성
            const preferredVoice = user.settings?.preferredVoice || '';
            
            // 음성 옵션 렌더링
            voices.forEach((voice, index) => {
                const isSelected = voice.short_name === preferredVoice;
                
                const voiceOption = document.createElement('div');
                voiceOption.className = `voice-option${isSelected ? ' selected' : ''}`;
                voiceOption.dataset.voice = voice.short_name;
                
                voiceOption.innerHTML = `
                    <div class="voice-icon"><i class="fas fa-microphone"></i></div>
                    <div class="voice-name">${voice.name}</div>
                    <div class="voice-gender">${voice.gender}</div>
                    <button class="voice-preview-btn"><i class="fas fa-play"></i></button>
                `;
                
                // 음성 선택 이벤트
                voiceOption.addEventListener('click', function(e) {
                    if (e.target.closest('.voice-preview-btn')) {
                        // 미리 듣기 버튼이 클릭된 경우
                        previewVoice(voice.short_name);
                    } else {
                        // 음성 옵션 선택
                        document.querySelectorAll('.voice-option').forEach(opt => {
                            opt.classList.remove('selected');
                        });
                        this.classList.add('selected');
                        savePreferences();
                    }
                });
                
                voiceContainer.appendChild(voiceOption);
            });
            
        } catch (error) {
            console.error('Error fetching voices:', error);
            document.getElementById('voice-selection').innerHTML = `
                <div class="alert alert-error">
                    <p>음성을 불러오는 중 오류가 발생했습니다.</p>
                </div>
            `;
        }
    }
    
    // 음성 미리 듣기
    async function previewVoice(voiceName) {
        try {
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/tts`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    text: '안녕하세요. 저는 당신의 음성 비서입니다.',
                    voice: voiceName,
                    rate: parseFloat(speechRateSlider.value),
                    pitch: parseFloat(speechPitchSlider.value)
                })
            });
            
            if (!response.ok) {
                throw new Error('Failed to generate voice preview');
            }
            
            const data = await response.json();
            
            // 오디오 재생
            const audio = new Audio(data.audio_url);
            audio.play();
            
        } catch (error) {
            console.error('Error previewing voice:', error);
            alert('음성 미리 듣기 중 오류가 발생했습니다.');
        }
    }
    
    // 프로필 업데이트
    async function updateProfile() {
        try {
            const firstName = document.getElementById('firstName').value;
            const lastName = document.getElementById('lastName').value;
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/profile`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    firstName,
                    lastName
                })
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '프로필 업데이트 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            
            // 로컬 스토리지 업데이트
            user = data.user;
            localStorage.setItem('user', JSON.stringify(user));
            
            // UI 업데이트
            updateUserInfo();
            
            alert('프로필이 성공적으로 업데이트되었습니다.');
            
        } catch (error) {
            console.error('Error updating profile:', error);
            alert(error.message);
        }
    }
    
    // 사용자 설정 업데이트
    async function updateUserSettings(settings) {
        try {
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/profile`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    settings,
                    preferredLanguage: settings.preferredLanguage,
                    preferredVoice: settings.preferredVoice
                })
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '설정 업데이트 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            
            // 로컬 스토리지 업데이트
            user = data.user;
            localStorage.setItem('user', JSON.stringify(user));
            
        } catch (error) {
            console.error('Error updating settings:', error);
            // 조용한 실패 - 사용자에게 알림 없음
        }
    }
    
    // 비밀번호 변경
    async function changePassword() {
        try {
            const currentPassword = document.getElementById('currentPassword').value;
            const newPassword = document.getElementById('newPassword').value;
            const confirmPassword = document.getElementById('confirmPassword').value;
            
            // 비밀번호 확인 검증
            if (newPassword !== confirmPassword) {
                alert('새 비밀번호와 확인이 일치하지 않습니다.');
                return;
            }
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/change-password`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    currentPassword,
                    newPassword
                })
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '비밀번호 변경 중 오류가 발생했습니다');
            }
            
            // 폼 초기화
            passwordForm.reset();
            
            // 성공 모달 표시
            passwordSuccessModal.style.display = 'block';
            
        } catch (error) {
            console.error('Error changing password:', error);
            alert(error.message);
        }
    }
    
    // 계정 비활성화
    async function deactivateAccount() {
        try {
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/deactivate`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    password: prompt('계정 비활성화를 확인하려면 비밀번호를 입력하세요:')
                })
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '계정 비활성화 중 오류가 발생했습니다');
            }
            
            // 로그아웃
            localStorage.removeItem('token');
            localStorage.removeItem('user');
            alert('계정이 성공적으로 비활성화되었습니다.');
            window.location.href = './login.html';
            
        } catch (error) {
            console.error('Error deactivating account:', error);
            alert(error.message);
        } finally {
            deactivateModal.style.display = 'none';
        }
    }
    
    // 계정 삭제
    async function deleteAccount() {
        try {
            const password = document.getElementById('delete-password').value;
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/users/${user.id}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    password
                })
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '계정 삭제 중 오류가 발생했습니다');
            }
            
            // 로그아웃
            localStorage.removeItem('token');
            localStorage.removeItem('user');
            alert('계정이 성공적으로 삭제되었습니다.');
            window.location.href = './index.html';
            
        } catch (error) {
            console.error('Error deleting account:', error);
            alert(error.message);
        } finally {
            deleteModal.style.display = 'none';
        }
    }
    
    // 모든 기기에서 로그아웃
    async function logoutAllDevices() {
        try {
            if (!confirm('모든 기기에서 로그아웃하시겠습니까?')) {
                return;
            }
            
            // API 요청
            const response = await fetch(`${API_BASE_URL}/api/auth/logout-all`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '로그아웃 중 오류가 발생했습니다');
            }
            
            // 현재 세션에서 로그아웃
            localStorage.removeItem('token');
            localStorage.removeItem('user');
            alert('모든 기기에서 로그아웃되었습니다.');
            window.location.href = './login.html';
            
        } catch (error) {
            console.error('Error logging out from all devices:', error);
            alert(error.message);
        }
    }
});