// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

document.addEventListener('DOMContentLoaded', function() {
    // 인증 확인
    const token = localStorage.getItem('token');
    if (!token) {
        window.location.href = './login.html';
        return;
    }
    
    // UI 요소
    const userNameElement = document.getElementById('user-name');
    const userAvatarElement = document.getElementById('user-avatar');
    const logoutBtn = document.getElementById('logout-btn');
    const themeToggle = document.getElementById('theme-toggle');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.getElementById('sidebar');
    
    const showAddFormBtn = document.getElementById('show-add-form');
    const reminderForm = document.getElementById('reminder-form');
    const addReminderForm = document.getElementById('add-reminder-form');
    const cancelReminderBtn = document.getElementById('cancel-reminder');
    
    const filterStatus = document.getElementById('filter-status');
    const filterType = document.getElementById('filter-type');
    const filterTime = document.getElementById('filter-time');
    const filterSort = document.getElementById('filter-sort');
    
    const todayReminders = document.getElementById('today-reminders');
    const upcomingReminders = document.getElementById('upcoming-reminders');
    const allReminders = document.getElementById('all-reminders');
    
    const reminderRelatedType = document.getElementById('reminder-related-type');
    const relatedItemGroup = document.getElementById('related-item-group');
    const relatedItemLabel = document.getElementById('related-item-label');
    const reminderRelatedId = document.getElementById('reminder-related-id');
    
    // 사용자 정보 로드
    let user = JSON.parse(localStorage.getItem('user')) || {};
    
    // 알림 데이터
    let reminders = [];
    let medications = [];
    let appointments = [];
    
    // 사용자 정보 표시
    userNameElement.textContent = user.firstName ? `${user.firstName} ${user.lastName || ''}` : user.username;
    if (user.firstName && user.lastName) {
        userAvatarElement.textContent = `${user.firstName.charAt(0)}${user.lastName.charAt(0)}`;
    } else if (user.username) {
        userAvatarElement.textContent = user.username.substring(0, 2).toUpperCase();
    }
    
    // 다크 모드 적용
    const currentTheme = localStorage.getItem('theme');
    if (currentTheme === 'dark') {
        document.body.setAttribute('data-theme', 'dark');
        themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
    }
    
    // Sidebar toggle on mobile
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    });
    
    // Theme toggle
    themeToggle.addEventListener('click', function() {
        if (document.body.getAttribute('data-theme') === 'dark') {
            document.body.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
        } else {
            document.body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        }
    });
    
    // Logout
    logoutBtn.addEventListener('click', function() {
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        window.location.href = './login.html';
    });
    
    // 알림 폼 표시
    showAddFormBtn.addEventListener('click', function() {
        // 폼 초기화
        addReminderForm.reset();
        document.getElementById('reminder-id').value = '';
        document.getElementById('form-title').textContent = '새 알림 추가';
        
        // 현재 시간 + 1시간을 기본값으로 설정
        const oneHourLater = new Date();
        oneHourLater.setHours(oneHourLater.getHours() + 1);
        
        // YYYY-MM-DDTHH:MM 형식으로 변환
        const formattedDateTime = oneHourLater.toISOString().slice(0, 16);
        document.getElementById('reminder-datetime').value = formattedDateTime;
        
        // 폼 표시
        reminderForm.style.display = 'block';
        
        // 페이지 상단으로 스크롤
        window.scrollTo({ top: 0, behavior: 'smooth' });
    });
    
    // 알림 추가 취소
    cancelReminderBtn.addEventListener('click', function() {
        reminderForm.style.display = 'none';
    });
    
    // 관련 항목 드롭다운 표시/숨김
    reminderRelatedType.addEventListener('change', function() {
        const relatedType = this.value;
        
        if (relatedType === 'other') {
            relatedItemGroup.style.display = 'none';
        } else {
            relatedItemGroup.style.display = 'block';
            
            // 라벨 업데이트
            if (relatedType === 'medication') {
                relatedItemLabel.textContent = '관련 약물';
                loadRelatedMedications();
            } else if (relatedType === 'appointment') {
                relatedItemLabel.textContent = '관련 예약';
                loadRelatedAppointments();
            }
        }
    });
    
    // 약물 데이터 로드 (관련 항목 드롭다운용)
    async function loadRelatedMedications() {
        try {
            // 이미 로드된 데이터가 있으면 재사용
            if (medications.length > 0) {
                populateMedicationDropdown();
                return;
            }
            
            const response = await fetch(`${API_BASE_URL}/api/medications?active=true`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('약물 정보를 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            medications = data.data || [];
            
            populateMedicationDropdown();
            
        } catch (error) {
            console.error('약물 로드 오류:', error);
            alert(error.message);
        }
    }
    
    // 예약 데이터 로드 (관련 항목 드롭다운용)
    async function loadRelatedAppointments() {
        try {
            // 이미 로드된 데이터가 있으면 재사용
            if (appointments.length > 0) {
                populateAppointmentDropdown();
                return;
            }
            
            const response = await fetch(`${API_BASE_URL}/api/appointments?upcoming=true`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('예약 정보를 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            appointments = data.data || [];
            
            populateAppointmentDropdown();
            
        } catch (error) {
            console.error('예약 로드 오류:', error);
            alert(error.message);
        }
    }
    
    // 약물 드롭다운 채우기
    function populateMedicationDropdown() {
        reminderRelatedId.innerHTML = '<option value="">-- 선택 --</option>';
        
        if (medications.length === 0) {
            reminderRelatedId.innerHTML += '<option value="" disabled>사용 가능한 약물이 없습니다</option>';
            return;
        }
        
        medications.forEach(medication => {
            reminderRelatedId.innerHTML += `<option value="${medication.id}">${medication.name} (${medication.dosage})</option>`;
        });
    }
    
    // 예약 드롭다운 채우기
    function populateAppointmentDropdown() {
        reminderRelatedId.innerHTML = '<option value="">-- 선택 --</option>';
        
        if (appointments.length === 0) {
            reminderRelatedId.innerHTML += '<option value="" disabled>사용 가능한 예약이 없습니다</option>';
            return;
        }
        
        appointments.forEach(appointment => {
            const appointmentDate = new Date(appointment.appointmentDate);
            const dateString = formatDate(appointmentDate, { month: 'short', day: 'numeric' });
            reminderRelatedId.innerHTML += `<option value="${appointment.id}">${appointment.doctorName} (${dateString})</option>`;
        });
    }
    
    // 알림 추가/수정 폼 제출
    addReminderForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const reminderId = document.getElementById('reminder-id').value;
        const title = document.getElementById('reminder-title').value;
        const description = document.getElementById('reminder-description').value || null;
        const dateTime = document.getElementById('reminder-datetime').value;
        const repeatType = document.getElementById('reminder-repeat').value;
        const relatedType = document.getElementById('reminder-related-type').value;
        const relatedId = relatedType !== 'other' ? document.getElementById('reminder-related-id').value : null;
        const notifyBefore = document.getElementById('reminder-notify').value;
        
        // 필수 필드 검증
        if (!title || !dateTime) {
            alert('필수 필드를 모두 작성해 주세요');
            return;
        }
        
        // 알림 데이터 생성
        const reminderData = {
            title,
            description,
            dateTime,
            repeatType,
            relatedType,
            relatedId: relatedId || null,
            notifyBefore: parseInt(notifyBefore),
            isCompleted: false,
            isActive: true
        };
        
        try {
            let response;
            
            if (reminderId) {
                // 알림 수정
                response = await fetch(`${API_BASE_URL}/api/reminders/${reminderId}`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(reminderData)
                });
            } else {
                // 새 알림 추가
                response = await fetch(`${API_BASE_URL}/api/reminders`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(reminderData)
                });
            }
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '알림 저장 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert(reminderId ? '알림이 성공적으로 업데이트되었습니다' : '알림이 성공적으로 추가되었습니다');
            
            // 폼 숨김
            reminderForm.style.display = 'none';
            
            // 알림 목록 새로고침
            loadReminders();
            
        } catch (error) {
            alert(error.message);
        }
    });
    
    // 필터 이벤트 리스너
    filterStatus.addEventListener('change', applyFilters);
    filterType.addEventListener('change', applyFilters);
    filterTime.addEventListener('change', applyFilters);
    filterSort.addEventListener('change', applyFilters);
    
    // 초기 데이터 로드
    loadReminders();
    
    // 전체 알림 데이터 가져오기
    async function loadReminders() {
        try {
            // 로딩 표시
            todayReminders.innerHTML = upcomingReminders.innerHTML = allReminders.innerHTML = `
                <div class="loading">
                    <div class="spinner"></div>
                </div>
            `;
            
            const response = await fetch(`${API_BASE_URL}/api/reminders`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('알림을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            reminders = data.data || [];
            console.log('알림 데이터:', reminders);
            
            // 알림 분류 및 표시
            categorizeAndDisplayReminders();
            
        } catch (error) {
            console.error('알림 로드 오류:', error);
            todayReminders.innerHTML = upcomingReminders.innerHTML = allReminders.innerHTML = `
                <div class="reminder-empty">
                    <p>오류: ${error.message}</p>
                </div>
            `;
        }
    }
    
    // 알림 분류 및 표시
    function categorizeAndDisplayReminders() {
        if (reminders.length === 0) {
            todayReminders.innerHTML = `<div class="reminder-empty">오늘 예정된 알림이 없습니다</div>`;
            upcomingReminders.innerHTML = `<div class="reminder-empty">다가오는 알림이 없습니다</div>`;
            allReminders.innerHTML = `<div class="reminder-empty">알림이 없습니다</div>`;
            return;
        }
        
        // 오늘 날짜
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        
        // 내일 날짜
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);
        
        // 일주일 후 날짜
        const nextWeek = new Date(today);
        nextWeek.setDate(nextWeek.getDate() + 7);
        
        // 알림 분류
        const todayRemindersList = [];
        const upcomingRemindersList = [];
        
        reminders.forEach(reminder => {
            const reminderDate = new Date(reminder.dateTime);
            const reminderDateOnly = new Date(reminderDate);
            reminderDateOnly.setHours(0, 0, 0, 0);
            
            // 오늘 알림
            if (reminderDateOnly.getTime() === today.getTime()) {
                todayRemindersList.push(reminder);
            }
            
            // 다가오는 알림 (내일부터 일주일 이내)
            if (reminderDateOnly >= tomorrow && reminderDateOnly < nextWeek) {
                upcomingRemindersList.push(reminder);
            }
        });
        
        // 알림 표시
        renderReminderList(todayReminders, todayRemindersList, 'today');
        renderReminderList(upcomingReminders, upcomingRemindersList, 'upcoming');
        
        // 필터 적용하여 전체 알림 표시
        applyFilters();
    }
    
    // 알림 목록 렌더링
    function renderReminderList(container, reminderList, listType) {
        if (reminderList.length === 0) {
            container.innerHTML = `<div class="reminder-empty">${listType === 'today' ? '오늘' : '다가오는'} 알림이 없습니다</div>`;
            return;
        }
        
        let html = '';
        
        // 날짜별로 정렬
        reminderList.sort((a, b) => new Date(a.dateTime) - new Date(b.dateTime));
        
        reminderList.forEach(reminder => {
            const reminderDate = new Date(reminder.dateTime);
            
            // 클래스 결정
            let reminderClass = '';
            if (reminder.isCompleted) {
                reminderClass = 'completed';
            } else {
                // 오늘 알림인지 확인
                const today = new Date();
                today.setHours(0, 0, 0, 0);
                
                const reminderDateOnly = new Date(reminderDate);
                reminderDateOnly.setHours(0, 0, 0, 0);
                
                if (reminderDateOnly.getTime() === today.getTime()) {
                    reminderClass = 'today';
                } else {
                    reminderClass = 'upcoming';
                }
            }
            
            // 관련 유형 표시
            let relatedTypeText = '';
            if (reminder.relatedType === 'medication') {
                relatedTypeText = '약물';
            } else if (reminder.relatedType === 'appointment') {
                relatedTypeText = '예약';
            } else {
                relatedTypeText = '기타';
            }
            
            // 반복 유형 표시
            let repeatTypeText = '';
            switch (reminder.repeatType) {
                case 'once':
                    repeatTypeText = '한 번';
                    break;
                case 'daily':
                    repeatTypeText = '매일';
                    break;
                case 'weekly':
                    repeatTypeText = '매주';
                    break;
                case 'monthly':
                    repeatTypeText = '매월';
                    break;
            }
            
            html += `
                <div class="reminder-item ${reminderClass}" data-id="${reminder.id}">
                    <div class="reminder-header">
                        <h3 class="reminder-title">${reminder.title}</h3>
                    </div>
                    
                    <div class="reminder-time">
                        <i class="far fa-clock"></i> ${formatDate(reminderDate)} ${formatTime(reminderDate)}
                    </div>
                    
                    ${reminder.description ? `
                        <div class="reminder-description">
                            ${reminder.description}
                        </div>
                    ` : ''}
                    
                    <div class="reminder-tags">
                        <span class="reminder-tag repeat">${repeatTypeText}</span>
                        <span class="reminder-tag related">${relatedTypeText}</span>
                    </div>
                    
                    <div class="reminder-actions">
                        <div class="reminder-complete">
                            <label>
                                <input type="checkbox" class="complete-checkbox" data-id="${reminder.id}" ${reminder.isCompleted ? 'checked' : ''}>
                                ${reminder.isCompleted ? '완료됨' : '완료 표시'}
                            </label>
                        </div>
                        
                        <div class="reminder-buttons">
                            <button class="edit-reminder" data-id="${reminder.id}" title="편집">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="delete-reminder" data-id="${reminder.id}" title="삭제">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
        
        // 편집 버튼 이벤트 리스너
        container.querySelectorAll('.edit-reminder').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                editReminder(id);
            });
        });
        
        // 삭제 버튼 이벤트 리스너
        container.querySelectorAll('.delete-reminder').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                deleteReminder(id);
            });
        });
        
        // 완료 체크박스 이벤트 리스너
        container.querySelectorAll('.complete-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const id = this.getAttribute('data-id');
                toggleReminderComplete(id, this.checked);
            });
        });
    }
    
    // 필터 적용
    function applyFilters() {
        // 필터 값 가져오기
        const status = filterStatus.value;
        const type = filterType.value;
        const time = filterTime.value;
        const sort = filterSort.value;
        
        // 필터링
        let filteredReminders = [...reminders];
        
        // 상태 필터
        if (status === 'active') {
            filteredReminders = filteredReminders.filter(reminder => !reminder.isCompleted);
        } else if (status === 'completed') {
            filteredReminders = filteredReminders.filter(reminder => reminder.isCompleted);
        }
        
        // 유형 필터
        if (type !== 'all') {
            filteredReminders = filteredReminders.filter(reminder => reminder.relatedType === type);
        }
        
        // 시간 필터
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);
        
        if (time === 'today') {
            filteredReminders = filteredReminders.filter(reminder => {
                const reminderDate = new Date(reminder.dateTime);
                reminderDate.setHours(0, 0, 0, 0);
                return reminderDate.getTime() === today.getTime();
            });
        } else if (time === 'upcoming') {
            filteredReminders = filteredReminders.filter(reminder => {
                const reminderDate = new Date(reminder.dateTime);
                reminderDate.setHours(0, 0, 0, 0);
                return reminderDate >= today;
            });
        } else if (time === 'past') {
            filteredReminders = filteredReminders.filter(reminder => {
                const reminderDate = new Date(reminder.dateTime);
                reminderDate.setHours(0, 0, 0, 0);
                return reminderDate < today;
            });
        }
        
        // 정렬
        if (sort === 'date-asc') {
            filteredReminders.sort((a, b) => new Date(a.dateTime) - new Date(b.dateTime));
        } else if (sort === 'date-desc') {
            filteredReminders.sort((a, b) => new Date(b.dateTime) - new Date(a.dateTime));
        } else if (sort === 'title-asc') {
            filteredReminders.sort((a, b) => a.title.localeCompare(b.title));
        } else if (sort === 'title-desc') {
            filteredReminders.sort((a, b) => b.title.localeCompare(a.title));
        }
        
        // 필터링된 알림 표시
        renderReminderList(allReminders, filteredReminders, 'all');
    }
    
    // 알림 편집
    async function editReminder(id) {
        try {
            const reminder = reminders.find(r => r.id.toString() === id.toString());
            
            if (!reminder) {
                throw new Error('알림을 찾을 수 없습니다');
            }
            
            // 폼에 데이터 채우기
            document.getElementById('reminder-id').value = reminder.id;
            document.getElementById('reminder-title').value = reminder.title;
            document.getElementById('reminder-description').value = reminder.description || '';
            
            // 날짜 및 시간 설정
            const dateTime = new Date(reminder.dateTime);
            // YYYY-MM-DDTHH:MM 형식으로 변환
            const formattedDateTime = dateTime.toISOString().slice(0, 16);
            document.getElementById('reminder-datetime').value = formattedDateTime;
            
            document.getElementById('reminder-repeat').value = reminder.repeatType;
            document.getElementById('reminder-related-type').value = reminder.relatedType;
            document.getElementById('reminder-notify').value = reminder.notifyBefore.toString();
            
            // 관련 항목 표시/숨김
            if (reminder.relatedType === 'other') {
                relatedItemGroup.style.display = 'none';
            } else {
                relatedItemGroup.style.display = 'block';
                
                if (reminder.relatedType === 'medication') {
                    relatedItemLabel.textContent = '관련 약물';
                    await loadRelatedMedications();
                } else if (reminder.relatedType === 'appointment') {
                    relatedItemLabel.textContent = '관련 예약';
                    await loadRelatedAppointments();
                }
                
                if (reminder.relatedId) {
                    document.getElementById('reminder-related-id').value = reminder.relatedId;
                }
            }
            
            // 폼 제목 변경
            document.getElementById('form-title').textContent = '알림 편집';
            
            // 폼 표시
            reminderForm.style.display = 'block';
            
            // 페이지 상단으로 스크롤
            window.scrollTo({ top: 0, behavior: 'smooth' });
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 알림 삭제
    async function deleteReminder(id) {
        // 확인 대화상자
        if (!confirm('이 알림을 삭제하시겠습니까? 이 작업은 취소할 수 없습니다.')) {
            return;
        }
        
        try {
            const response = await fetch(`${API_BASE_URL}/api/reminders/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('알림을 삭제하는 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert('알림이 성공적으로 삭제되었습니다');
            
            // 알림 목록 새로고침
            loadReminders();
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 알림 완료 상태 토글
    async function toggleReminderComplete(id, isCompleted) {
        try {
            const reminder = reminders.find(r => r.id.toString() === id.toString());
            
            if (!reminder) {
                throw new Error('알림을 찾을 수 없습니다');
            }
            
            const response = await fetch(`${API_BASE_URL}/api/reminders/${id}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    isCompleted
                })
            });
            
            if (!response.ok) {
                throw new Error('알림 상태를 변경하는 중 오류가 발생했습니다');
            }
            
            // 알림 목록 새로고침
            loadReminders();
            
        } catch (error) {
            alert(error.message);
            console.error('완료 상태 변경 오류:', error);
        }
    }
    
    // 날짜 형식 함수
    function formatDate(date, options = {}) {
        const defaultOptions = {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            weekday: 'long'
        };
        
        const mergedOptions = {...defaultOptions, ...options};
        
        return date.toLocaleDateString('ko-KR', mergedOptions);
    }
    
    // 시간 형식 함수
    function formatTime(date) {
        return date.toLocaleTimeString('ko-KR', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: true
        });
    }
});