// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

document.addEventListener('DOMContentLoaded', function() {
    // 인증 확인
    const token = localStorage.getItem('token');
    if (!token) {
        window.location.href = './login.html';
        return;
    }
    
    // UI 요소
    const userNameElement = document.getElementById('user-name');
    const userAvatarElement = document.getElementById('user-avatar');
    const logoutBtn = document.getElementById('logout-btn');
    const themeToggle = document.getElementById('theme-toggle');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.getElementById('sidebar');
    
    const medicationsList = document.getElementById('medications-list');
    const addMedicationBtn = document.getElementById('show-add-form');
    const medicationForm = document.getElementById('medication-form');
    const addMedicationForm = document.getElementById('add-medication-form');
    const cancelMedicationBtn = document.getElementById('cancel-medication');
    const medFrequencySelect = document.getElementById('med-frequency');
    const customFrequencyDiv = document.getElementById('custom-frequency');
    const activeOnlyCheckbox = document.getElementById('active-only');
    const sortBySelect = document.getElementById('sort-by');
    const searchInput = document.getElementById('search-medications');
    
    // 사용자 정보 로드
    let user = JSON.parse(localStorage.getItem('user')) || {};
    
    // 사용자 정보 표시
    userNameElement.textContent = user.firstName ? `${user.firstName} ${user.lastName || ''}` : user.username;
    if (user.firstName && user.lastName) {
        userAvatarElement.textContent = `${user.firstName.charAt(0)}${user.lastName.charAt(0)}`;
    } else if (user.username) {
        userAvatarElement.textContent = user.username.substring(0, 2).toUpperCase();
    }
    
    // 다크 모드 적용
    const currentTheme = localStorage.getItem('theme');
    if (currentTheme === 'dark') {
        document.body.setAttribute('data-theme', 'dark');
        themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
    }
    
    // Sidebar toggle on mobile
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    });
    
    // Theme toggle
    themeToggle.addEventListener('click', function() {
        if (document.body.getAttribute('data-theme') === 'dark') {
            document.body.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
        } else {
            document.body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        }
    });
    
    // Logout
    logoutBtn.addEventListener('click', function() {
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        window.location.href = './login.html';
    });
    
    // 약품 목록 로드
    loadMedications();
    
    // 약품 추가 폼 표시
    addMedicationBtn.addEventListener('click', function() {
        // 폼 초기화
        addMedicationForm.reset();
        document.getElementById('medication-id').value = '';
        document.getElementById('form-title').textContent = '새 약품 추가';
        
        // 오늘 날짜를 시작일로 설정
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('med-start-date').value = today;
        
        // 폼 표시
        medicationForm.style.display = 'block';
        
        // 페이지 상단으로 스크롤
        window.scrollTo({ top: 0, behavior: 'smooth' });
    });
    
    // 약품 추가 취소
    cancelMedicationBtn.addEventListener('click', function() {
        medicationForm.style.display = 'none';
    });
    
    // 맞춤 빈도 표시/숨김
    medFrequencySelect.addEventListener('change', function() {
        if (this.value === '기타') {
            customFrequencyDiv.style.display = 'block';
        } else {
            customFrequencyDiv.style.display = 'none';
        }
    });
    
    // 약품 추가/수정 폼 제출
    addMedicationForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const medicationId = document.getElementById('medication-id').value;
        const name = document.getElementById('med-name').value;
        const dosage = document.getElementById('med-dosage').value;
        let frequency = document.getElementById('med-frequency').value;
        const startDate = document.getElementById('med-start-date').value;
        const endDate = document.getElementById('med-end-date').value || null;
        const doctorName = document.getElementById('med-doctor').value || null;
        const purpose = document.getElementById('med-purpose').value || null;
        const sideEffects = document.getElementById('med-side-effects').value || null;
        const notes = document.getElementById('med-notes').value || null;
        
        // 맞춤 빈도 처리
        if (frequency === '기타') {
            frequency = document.getElementById('med-custom-frequency').value;
        }
        
        // 필수 필드 검증
        if (!name || !dosage || !frequency || !startDate) {
            alert('필수 필드를 모두 작성해 주세요');
            return;
        }
        
        // 약품 데이터 생성
        const medicationData = {
            name,
            dosage,
            frequency,
            startDate,
            endDate,
            doctorName,
            purpose,
            sideEffects,
            notes
        };
        
        try {
            let response;
            
            if (medicationId) {
                // 약품 수정 (기본 URL 접두사 추가)
                response = await fetch(`${API_BASE_URL}/api/medications/${medicationId}`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(medicationData)
                });
            } else {
                // 새 약품 추가 (기본 URL 접두사 추가)
                response = await fetch(`${API_BASE_URL}/api/medications`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(medicationData)
                });
            }
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '약품 저장 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert(medicationId ? '약품이 성공적으로 업데이트되었습니다' : '약품이 성공적으로 추가되었습니다');
            
            // 폼 숨김
            medicationForm.style.display = 'none';
            
            // 약품 목록 새로고침
            loadMedications();
            
        } catch (error) {
            alert(error.message);
        }
    });
    
    // 활성화 필터 변경
    activeOnlyCheckbox.addEventListener('change', function() {
        loadMedications();
    });
    
    // 정렬 변경
    sortBySelect.addEventListener('change', function() {
        loadMedications();
    });
    
    // 검색 입력
    searchInput.addEventListener('input', function() {
        loadMedications();
    });
    
    // 약품 목록 로드 함수
    async function loadMedications() {
        // 로딩 표시
        medicationsList.innerHTML = `
            <div class="loading">
                <div class="spinner"></div>
            </div>
        `;
        
        // 필터 및 정렬 설정
        const activeOnly = activeOnlyCheckbox.checked;
        const sortBy = sortBySelect.value;
        const searchTerm = searchInput.value.trim();
        
        // API URL 구성
        let url = '/api/medications?';
        if (activeOnly) {
            url += 'active=true&';
        }
        if (searchTerm) {
            url += `search=${encodeURIComponent(searchTerm)}&`;
        }
        url += `sort=${sortBy}`;
        
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}${url}`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('약품을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            
            // 약품 목록 렌더링
            renderMedications(data.data || []);
            
        } catch (error) {
            medicationsList.innerHTML = `<p>오류: ${error.message}</p>`;
        }
    }
    
    // 약품 목록 렌더링 함수
    function renderMedications(medications) {
        if (medications.length === 0) {
            medicationsList.innerHTML = '<p>등록된 약품이 없습니다</p>';
            return;
        }
        
        let html = '';
        
        medications.forEach(med => {
            html += `
                <div class="medication-card ${med.is_active ? '' : 'inactive'}">
                    <div class="medication-header">
                        <h3>${med.name}</h3>
                        <div class="medication-actions">
                            <button class="edit-medication" data-id="${med.id}" title="수정">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="delete-medication" data-id="${med.id}" data-name="${med.name}" title="삭제">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    <div class="medication-details">
                        <p><strong>복용량:</strong> ${med.dosage}</p>
                        <p><strong>빈도:</strong> ${med.frequency}</p>
                        <p><strong>시작일:</strong> ${formatDate(med.start_date)}</p>
                        ${med.end_date ? `<p><strong>종료일:</strong> ${formatDate(med.end_date)}</p>` : ''}
                        ${med.doctor_name ? `<p><strong>의사:</strong> ${med.doctor_name}</p>` : ''}
                        ${med.purpose ? `<p><strong>용도:</strong> ${med.purpose}</p>` : ''}
                    </div>
                    <div class="medication-reminders">
                        <h4>리마인더</h4>
                        <div class="reminder-list" id="reminder-list-${med.id}">
                            <div class="loading">
                                <div class="spinner"></div>
                            </div>
                        </div>
                        <button class="add-reminder" data-id="${med.id}" data-name="${med.name}">
                            <i class="fas fa-bell"></i> 리마인더 추가
                        </button>
                    </div>
                    <div class="medication-toggle">
                        <label class="switch">
                            <input type="checkbox" class="active-toggle" data-id="${med.id}" ${med.is_active ? 'checked' : ''}>
                            <span class="slider round"></span>
                        </label>
                        <span>${med.is_active ? '활성' : '비활성'}</span>
                    </div>
                </div>
            `;
        });
        
        medicationsList.innerHTML = html;
        
        // 약품별 리마인더 로드
        medications.forEach(med => {
            loadReminders(med.id);
        });
        
        // 수정 버튼 이벤트 리스너
        document.querySelectorAll('.edit-medication').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                editMedication(id);
            });
        });
        
        // 삭제 버튼 이벤트 리스너
        document.querySelectorAll('.delete-medication').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const name = this.getAttribute('data-name');
                deleteMedication(id, name);
            });
        });
        
        // 활성화 토글 이벤트 리스너
        document.querySelectorAll('.active-toggle').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const id = this.getAttribute('data-id');
                toggleMedicationStatus(id, this.checked);
            });
        });
        
        // 리마인더 추가 버튼 이벤트 리스너
        document.querySelectorAll('.add-reminder').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const name = this.getAttribute('data-name');
                addReminder(id, name);
            });
        });
    }
    
    // 약품 수정 함수
    async function editMedication(id) {
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/${id}`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('약품 정보를 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const medication = data.data;
            
            // 폼에 데이터 채우기
            document.getElementById('medication-id').value = medication.id;
            document.getElementById('med-name').value = medication.name;
            document.getElementById('med-dosage').value = medication.dosage;
            
            // 빈도 설정
            const frequencySelect = document.getElementById('med-frequency');
            const predefinedFrequencies = ['하루에 한 번', '하루에 두 번', '하루에 세 번', '필요할 때마다'];
            
            if (predefinedFrequencies.includes(medication.frequency)) {
                frequencySelect.value = medication.frequency;
                customFrequencyDiv.style.display = 'none';
            } else {
                frequencySelect.value = '기타';
                document.getElementById('med-custom-frequency').value = medication.frequency;
                customFrequencyDiv.style.display = 'block';
            }
            
            // 날짜 설정
            document.getElementById('med-start-date').value = medication.start_date.split('T')[0];
            if (medication.end_date) {
                document.getElementById('med-end-date').value = medication.end_date.split('T')[0];
            } else {
                document.getElementById('med-end-date').value = '';
            }
            
            // 기타 필드
            document.getElementById('med-doctor').value = medication.doctor_name || '';
            document.getElementById('med-purpose').value = medication.purpose || '';
            document.getElementById('med-side-effects').value = medication.side_effects || '';
            document.getElementById('med-notes').value = medication.notes || '';
            
            // 폼 제목 변경
            document.getElementById('form-title').textContent = '약품 편집';
            
            // 폼 표시
            medicationForm.style.display = 'block';
            
            // 페이지 상단으로 스크롤
            window.scrollTo({ top: 0, behavior: 'smooth' });
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 약품 삭제 함수
    async function deleteMedication(id, name) {
        // 확인 대화상자
        if (!confirm(`"${name}" 약품을 삭제하시겠습니까? 이 작업은 취소할 수 없습니다.`)) {
            return;
        }
        
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('약품을 삭제하는 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert('약품이 성공적으로 삭제되었습니다');
            
            // 약품 목록 새로고침
            loadMedications();
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 약품 활성화 상태 토글 함수
    async function toggleMedicationStatus(id, isActive) {
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/${id}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    isActive
                })
            });
            
            if (!response.ok) {
                throw new Error('약품 상태를 변경하는 중 오류가 발생했습니다');
            }
            
            // 약품 목록 새로고침
            loadMedications();
            
        } catch (error) {
            alert(error.message);
            
            // 체크박스 상태 원복 (실패 시)
            const checkbox = document.querySelector(`.active-toggle[data-id="${id}"]`);
            if (checkbox) {
                checkbox.checked = !isActive;
            }
        }
    }
    
    // 리마인더 로드 함수
    async function loadReminders(medicationId) {
        const reminderListElement = document.getElementById(`reminder-list-${medicationId}`);
        
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/${medicationId}/reminders`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('리마인더를 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const reminders = data.data || [];
            
            // 리마인더 목록 렌더링
            if (reminders.length === 0) {
                reminderListElement.innerHTML = '<p>등록된 리마인더가 없습니다</p>';
            } else {
                let html = '';
                
                reminders.forEach(reminder => {
                    const reminderTime = reminder.reminder_time.substring(0, 5); // HH:MM 형식
                    
                    html += `
                        <div class="reminder-item">
                            <div class="reminder-time">
                                ${reminderTime}
                                ${reminder.is_daily ? '(매일)' : ''}
                            </div>
                            <div class="reminder-status">
                                <div class="reminder-toggle">
                                    <input type="checkbox" id="reminder-${reminder.id}" class="reminder-toggle-input" data-id="${reminder.id}" ${reminder.is_completed ? 'checked' : ''}>
                                    <label for="reminder-${reminder.id}"></label>
                                </div>
                                <button class="delete-reminder" data-id="${reminder.id}" title="삭제">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                        </div>
                    `;
                });
                
                reminderListElement.innerHTML = html;
                
                // 리마인더 토글 이벤트 리스너
                reminderListElement.querySelectorAll('.reminder-toggle-input').forEach(checkbox => {
                    checkbox.addEventListener('change', function() {
                        const reminderId = this.getAttribute('data-id');
                        toggleReminderStatus(reminderId, this.checked);
                    });
                });
                
                // 리마인더 삭제 이벤트 리스너
                reminderListElement.querySelectorAll('.delete-reminder').forEach(button => {
                    button.addEventListener('click', function() {
                        const reminderId = this.getAttribute('data-id');
                        deleteReminder(reminderId, medicationId);
                    });
                });
            }
            
        } catch (error) {
            reminderListElement.innerHTML = `<p>오류: ${error.message}</p>`;
        }
    }
    
    // 리마인더 추가 함수
    function addReminder(medicationId, medicationName) {
        // 여기서는 간단한 알림 창으로 구현
        // 실제 구현에서는 모달이나 폼을 사용할 수 있음
        const time = prompt(`"${medicationName}" 약품의 리마인더 시간을 입력하세요 (HH:MM 형식):`, '09:00');
        
        if (!time) return;
        
        // 간단한 시간 형식 검증
        const timeRegex = /^([01]\d|2[0-3]):([0-5]\d)$/;
        if (!timeRegex.test(time)) {
            alert('시간은 HH:MM 형식으로 입력해주세요 (예: 09:00)');
            return;
        }
        
        // 매일 반복 여부
        const isDaily = confirm('이 리마인더를 매일 반복하시겠습니까?');

        // 리마인더 생성 API 호출
        createReminder(medicationId, time, isDaily);
    }
    
    // 리마인더 생성 API 호출 함수
    async function createReminder(medicationId, reminderTime, isDaily) {
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/${medicationId}/reminders`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    reminderTime,
                    isDaily
                })
            });
            
            if (!response.ok) {
                throw new Error('리마인더를 추가하는 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert('리마인더가 성공적으로 추가되었습니다');
            
            // 리마인더 목록 새로고침
            loadReminders(medicationId);
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 리마인더 상태 토글 함수
    async function toggleReminderStatus(reminderId, isCompleted) {
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/reminders/${reminderId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    isCompleted
                })
            });
            
            if (!response.ok) {
                throw new Error('리마인더 상태를 변경하는 중 오류가 발생했습니다');
            }
            
            // 특별한 처리 필요 없음 - 체크박스가 상태를 반영함
            
        } catch (error) {
            alert(error.message);
            
            // 체크박스 상태 원복 (실패 시)
            const checkbox = document.getElementById(`reminder-${reminderId}`);
            if (checkbox) {
                checkbox.checked = !isCompleted;
            }
        }
    }
    
    // 리마인더 삭제 함수
    async function deleteReminder(reminderId, medicationId) {
        // 확인 대화상자
        if (!confirm('이 리마인더를 삭제하시겠습니까?')) {
            return;
        }
        
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/medications/reminders/${reminderId}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('리마인더를 삭제하는 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert('리마인더가 성공적으로 삭제되었습니다');
            
            // 리마인더 목록 새로고침
            loadReminders(medicationId);
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 날짜 형식 함수
    function formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('ko-KR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    }
});